/*----------------------------------------------------------------------------\
|                                Slider                                       |
|-----------------------------------------------------------------------------|
|                         Created by Erik Arvidsson                           |
|                  (http://webfx.eae.net/contact.html#erik)                   |
|                      For WebFX (http://webfx.eae.net/)                      |
|-----------------------------------------------------------------------------|
| A  slider  control that  degrades  to an  input control  for non  supported |
| browsers.                                                                   |
|-----------------------------------------------------------------------------|
|    Copyright (c) 2002, 2003, 2006 Erik Arvidsson                            |
|-----------------------------------------------------------------------------|
| Licensed under the Apache License, Version 2.0 (the "License"); you may not |
| use this file except in compliance with the License.  You may obtain a copy |
| of the License at http://www.apache.org/licenses/LICENSE-2.0                |
| - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - |
| Unless  required  by  applicable law or  agreed  to  in  writing,  software |
| distributed under the License is distributed on an  "AS IS" BASIS,  WITHOUT |
| WARRANTIES OR  CONDITIONS OF ANY KIND,  either express or implied.  See the |
| License  for the  specific language  governing permissions  and limitations |
| under the License.                                                          |
|-----------------------------------------------------------------------------|
| 2002-10-14 | Original version released                                      |
| 2006-05-28 | Changed license to Apache Software License 2.0.                |
|-----------------------------------------------------------------------------|
| 2010-12-10 | included range.js in the same file                <by a.r.c>   |
| 2012-04-25 | major rewrite, use Drag object                    <by a.r.c>   |
| 2012-04-28 | pass inital value for Range                       <by a.r.c>   |
| 2012-11-26 | re-write to use Drag-06.js for event handling     <by a.r.c>   |
| 2013-08-09 | re-write                                          <by a.r.c>   |
\----------------------------------------------------------------------------*/

function Slider(sliderPanelId, sOrientation, val, callBackFn)
{
  /* slider is made of 3 components:
    slider panel which is defined in the HTML
    slider line, the slot where the handle slides
    slider handle, the bit that you grab and drag */
  var savThis = this,
	    orientation = (sOrientation == "vertical")? "vertical": "horizontal",
      callBack = (typeof callBackFn == "function")? callBackFn: null,
      panel, line, handle,
    	w, h, hw, hh, lw, lh,
      sliderDragData;               // payload

  panel = document.getElementById(sliderPanelId);
  // add class name tag to class name
  panel.className = orientation + " dynamic-slider-control " + panel.className;
  // create line
  line = document.createElement("DIV");
  line.className = "line";
  line.appendChild(document.createElement("DIV"));
  panel.appendChild(line);
  // create handle
  handle = document.createElement("DIV");
  handle.className = "handle";
  handle.appendChild(document.createElement("DIV"));
  handle.firstChild.appendChild(document.createTextNode(String.fromCharCode(160)));
  panel.appendChild(handle);
	w = panel.offsetWidth;
	h = panel.offsetHeight;
	hw = handle.offsetWidth;
	hh = handle.offsetHeight;
	lw = line.offsetWidth;
	lh = line.offsetHeight;

  function getCsrPos(event)
  {
    var e = event||window.event,
        doc,
        csrPos = {};
    // now find coords of cursor relative to the page
    if (e.pageX == null)
    {
      // works for IE 6, 7, 8
      doc = (document.documentElement && document.documentElement.scrollLeft != null) ?
              document.documentElement : document.body;
      csrPos.x = e.clientX + doc.scrollLeft;
      csrPos.y = e.clientY + doc.scrollTop;
    }
    else  // all other browsers
    {
      csrPos.x = e.pageX;
      csrPos.y = e.pageY;
    }
    return csrPos;
  };

  function grab(e)
  {
    var evt = e||window.event,
        csrPos;

    document.onmouseup = function()
    {
      document.onmouseup = null;
      document.onmousedown = null;
      document.onmousemove = null;
    };
    csrPos = getCsrPos(evt);     // update mouse pos to pass to the owner
    sliderDragData =  { dragStartX: csrPos.x,
                        dragStartY: csrPos.y,
                        startValue: savThis.getValue(),
                        dx:	csrPos.x - handle.offsetLeft,  // subtract current handle pos on slider
                        dy:	csrPos.x - handle.offsetTop };
    document.onmousemove = drag;

    if (evt.preventDefault) // Prevent default browser action (W3C)
    {
      evt.preventDefault();
    }
    else                  // A shortcut for stopping the browser action in IE
    {
      window.event.returnValue = false;
    }
    return false;
	};

  function drag(e)
  {
    var csrPos = getCsrPos(e),     // update mouse pos to pass to the owner
  	    boundSize = savThis.getMaximum() - savThis.getMinimum(),
  	    size, pos, reset;

  	if (orientation == "horizontal")
    {
  		size = panel.offsetWidth - handle.offsetWidth;
  		pos = csrPos.x - sliderDragData.dx;  // change in mouseX since grab + handle pos on slider at grab
  		reset = Math.abs(csrPos.y - sliderDragData.dragStartY) > 100;  // dragging too far, reset
  	}
  	else
    {
  		size = panel.offsetHeight - handle.offsetHeight;
      // to calculate y pos note slider is +ve up screen!
  		pos = panel.offsetHeight - handle.offsetHeight - (mouseY - sliderDragData.dy);
  		reset = Math.abs(csrPos.x - sliderDragData.dragStartX) > 100;     // dragging too far, reset
  	}
  	savThis.setValue(reset ? sliderDragData.startValue :savThis.getMinimum() + boundSize * pos / size);
  	return false;
  };

  this.recalculate = function()
  {
  	// this assumes a border-box layout
  	if (orientation == "horizontal")
    {
  		handle.style.left = (w - hw) * (this.getValue() - this.getMinimum()) /
  			(this.getMaximum() - this.getMinimum()) + "px";
  		handle.style.top = (h - hh) / 2 + "px";

  		line.style.top = (h - lh) / 2 + "px";
  		line.style.left = hw / 2 + "px";
  		//line.style.right = hw / 2 + "px";
  		line.style.width = Math.max(0, w - hw - 2)+ "px";
  		line.firstChild.style.width = Math.max(0, w - hw - 4)+ "px";
  	}
  	else
    {
  		handle.style.left = (w - hw) / 2 + "px";
  		handle.style.top = h - hh - (h - hh) * (this.getValue() - this.getMinimum()) /
  			(this.getMaximum() - this.getMinimum()) + "px";

  		line.style.left = (w - lw) / 2 + "px";
  		line.style.top = hh / 2 + "px";
  		line.style.height = Math.max(0, h - hh - 2) + "px";	//hard coded border width
  		//line.style.bottom = hh / 2 + "px";
  		line.firstChild.style.height = Math.max(0, h - hh - 4) + "px";	//hard coded border width
  	}
  };

	this._range = new Range(val);

	this._range.onchange = function()
  {
		savThis.recalculate();
		if (callBack != null)
		  callBack();
	};

  // set up the drag handler
  handle.onmousedown = grab;

  this.recalculate();   // position the handle to default
}

Slider.prototype.setValue = function(v)
{
	this._range.setValue(v);
  this.recalculate();       // bugfix: setting initial value needs recalc
};

Slider.prototype.getValue = function()
{
	return this._range.getValue();
};

Slider.prototype.setMinimum = function(v)
{
	this._range.setMinimum(v);
};

Slider.prototype.getMinimum = function()
{
	return this._range.getMinimum();
};

Slider.prototype.setMaximum = function(v)
{
	this._range.setMaximum(v);
};

Slider.prototype.getMaximum = function()
{
	return this._range.getMaximum();
};

function Range(value)
{
	this._minimum = 0;
	this._maximum = 100;
	this._extent = 0;

	value = Math.round(parseFloat(value));
	if (isNaN(value)) return;

  if (value + this._extent > this._maximum)
  	this._value = this._maximum - this._extent;
  else if (value < this._minimum)
  	this._value = this._minimum;
  else
  	this._value = value;

	this._isChanging = false;
}

Range.prototype.setValue = function(value)
{
	value = Math.round(parseFloat(value));
	if (isNaN(value)) return;
	if (this._value != value)
  {
		if (value + this._extent > this._maximum)
			this._value = this._maximum - this._extent;
		else if (value < this._minimum)
			this._value = this._minimum;
		else
			this._value = value;
		if (!this._isChanging && typeof this.onchange == "function")
			 this.onchange();
	}
};

Range.prototype.getValue = function()
{
	return this._value;
};

Range.prototype.setExtent = function(extent)
{
	if (this._extent != extent)
  {
		if (extent < 0)
			this._extent = 0;
		else if (this._value + extent > this._maximum)
			this._extent = this._maximum - this._value;
		else
			this._extent = extent;
		if (!this._isChanging && typeof this.onchange == "function")
			this.onchange();
	}
};

Range.prototype.getExtent = function()
{
	return this._extent;
};

Range.prototype.setMinimum = function(minimum)
{
	if (this._minimum != minimum)
  {
		var oldIsChanging = this._isChanging;
		this._isChanging = true;

		this._minimum = minimum;

		if (minimum > this._value)
			this.setValue(minimum);
		if (minimum > this._maximum)
    {
			this._extent = 0;
			this.setMaximum(minimum);
			this.setValue(minimum)
		}
		if (minimum + this._extent > this._maximum)
			this._extent = this._maximum - this._minimum;

		this._isChanging = oldIsChanging;
		if (!this._isChanging && typeof this.onchange == "function")
			this.onchange();
	}
};

Range.prototype.getMinimum = function()
{
	return this._minimum;
};

Range.prototype.setMaximum = function(maximum)
{
	if (this._maximum != maximum)
  {
		var oldIsChanging = this._isChanging;
		this._isChanging = true;

		this._maximum = maximum;

		if (maximum < this._value)
			this.setValue(maximum - this._extent);
		if (maximum < this._minimum)
    {
			this._extent = 0;
			this.setMinimum(maximum);
			this.setValue(this._maximum);
		}
		if (maximum < this._minimum + this._extent)
			this._extent = this._maximum - this._minimum;
		if (maximum < this._value + this._extent)
			this._extent = this._maximum - this._value;

		this._isChanging = oldIsChanging;
		if (!this._isChanging && typeof this.onchange == "function")
			this.onchange();
	}
};

Range.prototype.getMaximum = function()
{
	return this._maximum;
};
